/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/// <reference types="@types/dom-view-transitions" />
import { DOCUMENT } from '@angular/common';
import { afterNextRender, InjectionToken, NgZone, runInInjectionContext } from '@angular/core';
export const CREATE_VIEW_TRANSITION = new InjectionToken(ngDevMode ? 'view transition helper' : '');
export const VIEW_TRANSITION_OPTIONS = new InjectionToken(ngDevMode ? 'view transition options' : '');
/**
 * A helper function for using browser view transitions. This function skips the call to
 * `startViewTransition` if the browser does not support it.
 *
 * @returns A Promise that resolves when the view transition callback begins.
 */
export function createViewTransition(injector, from, to) {
    const transitionOptions = injector.get(VIEW_TRANSITION_OPTIONS);
    const document = injector.get(DOCUMENT);
    // Create promises outside the Angular zone to avoid causing extra change detections
    return injector.get(NgZone).runOutsideAngular(() => {
        if (!document.startViewTransition || transitionOptions.skipNextTransition) {
            transitionOptions.skipNextTransition = false;
            return Promise.resolve();
        }
        let resolveViewTransitionStarted;
        const viewTransitionStarted = new Promise((resolve) => {
            resolveViewTransitionStarted = resolve;
        });
        const transition = document.startViewTransition(() => {
            resolveViewTransitionStarted();
            // We don't actually update dom within the transition callback. The resolving of the above
            // promise unblocks the Router navigation, which synchronously activates and deactivates
            // routes (the DOM update). This view transition waits for the next change detection to
            // complete (below), which includes the update phase of the routed components.
            return createRenderPromise(injector);
        });
        const { onViewTransitionCreated } = transitionOptions;
        if (onViewTransitionCreated) {
            runInInjectionContext(injector, () => onViewTransitionCreated({ transition, from, to }));
        }
        return viewTransitionStarted;
    });
}
/**
 * Creates a promise that resolves after next render.
 */
function createRenderPromise(injector) {
    return new Promise(resolve => {
        afterNextRender(resolve, { injector });
    });
}
//# sourceMappingURL=data:application/json;base64,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