/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { consumerAfterComputation, consumerBeforeComputation, consumerDestroy, consumerMarkDirty, consumerPollProducersForChange, isInNotificationPhase, REACTIVE_NODE, SIGNAL } from './graph';
export function createWatch(fn, schedule, allowSignalWrites) {
    const node = Object.create(WATCH_NODE);
    if (allowSignalWrites) {
        node.consumerAllowSignalWrites = true;
    }
    node.fn = fn;
    node.schedule = schedule;
    const registerOnCleanup = (cleanupFn) => {
        node.cleanupFn = cleanupFn;
    };
    function isWatchNodeDestroyed(node) {
        return node.fn === null && node.schedule === null;
    }
    function destroyWatchNode(node) {
        if (!isWatchNodeDestroyed(node)) {
            consumerDestroy(node); // disconnect watcher from the reactive graph
            node.cleanupFn();
            // nullify references to the integration functions to mark node as destroyed
            node.fn = null;
            node.schedule = null;
            node.cleanupFn = NOOP_CLEANUP_FN;
        }
    }
    const run = () => {
        if (node.fn === null) {
            // trying to run a destroyed watch is noop
            return;
        }
        if (isInNotificationPhase()) {
            throw new Error(`Schedulers cannot synchronously execute watches while scheduling.`);
        }
        node.dirty = false;
        if (node.hasRun && !consumerPollProducersForChange(node)) {
            return;
        }
        node.hasRun = true;
        const prevConsumer = consumerBeforeComputation(node);
        try {
            node.cleanupFn();
            node.cleanupFn = NOOP_CLEANUP_FN;
            node.fn(registerOnCleanup);
        }
        finally {
            consumerAfterComputation(node, prevConsumer);
        }
    };
    node.ref = {
        notify: () => consumerMarkDirty(node),
        run,
        cleanup: () => node.cleanupFn(),
        destroy: () => destroyWatchNode(node),
        [SIGNAL]: node,
    };
    return node.ref;
}
const NOOP_CLEANUP_FN = () => { };
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
const WATCH_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        consumerIsAlwaysLive: true,
        consumerAllowSignalWrites: false,
        consumerMarkedDirty: (node) => {
            if (node.schedule !== null) {
                node.schedule(node.ref);
            }
        },
        hasRun: false,
        cleanupFn: NOOP_CLEANUP_FN,
    };
})();
//# sourceMappingURL=data:application/json;base64,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