/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Inside the body of a repeater, certain context variables (such as `$first`) are ambiently
 * available. This phase finds those variable usages, and replaces them with the appropriate
 * expression.
 */
export function generateRepeaterDerivedVars(job) {
    const repeaters = new Map();
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            if (op.kind === ir.OpKind.RepeaterCreate) {
                repeaters.set(op.xref, op);
            }
        }
    }
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            ir.transformExpressionsInOp(op, expr => {
                if (!(expr instanceof ir.DerivedRepeaterVarExpr)) {
                    return expr;
                }
                const repeaterOp = repeaters.get(expr.xref);
                switch (expr.identity) {
                    case ir.DerivedRepeaterVarIdentity.First:
                        return new o.BinaryOperatorExpr(o.BinaryOperator.Identical, new ir.LexicalReadExpr(repeaterOp.varNames.$index), o.literal(0));
                    case ir.DerivedRepeaterVarIdentity.Last:
                        return new o.BinaryOperatorExpr(o.BinaryOperator.Identical, new ir.LexicalReadExpr(repeaterOp.varNames.$index), new o.BinaryOperatorExpr(o.BinaryOperator.Minus, new ir.LexicalReadExpr(repeaterOp.varNames.$count), o.literal(1)));
                    case ir.DerivedRepeaterVarIdentity.Even:
                        return new o.BinaryOperatorExpr(o.BinaryOperator.Identical, new o.BinaryOperatorExpr(o.BinaryOperator.Modulo, new ir.LexicalReadExpr(repeaterOp.varNames.$index), o.literal(2)), o.literal(0));
                    case ir.DerivedRepeaterVarIdentity.Odd:
                        return new o.BinaryOperatorExpr(o.BinaryOperator.NotIdentical, new o.BinaryOperatorExpr(o.BinaryOperator.Modulo, new ir.LexicalReadExpr(repeaterOp.varNames.$index), o.literal(2)), o.literal(0));
                }
            }, ir.VisitorContextFlag.None);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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