/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../../output/output_ast';
import { Namespace, OpKind } from '../enums';
import { SlotHandle } from '../handle';
import { OpList } from '../operations';
import { TRAIT_CONSUMES_SLOT } from '../traits';
import { NEW_OP } from './shared';
/**
 * The set of OpKinds that represent the creation of an element or container
 */
const elementContainerOpKinds = new Set([
    OpKind.Element, OpKind.ElementStart, OpKind.Container, OpKind.ContainerStart, OpKind.Template,
    OpKind.RepeaterCreate
]);
/**
 * Checks whether the given operation represents the creation of an element or container.
 */
export function isElementOrContainerOp(op) {
    return elementContainerOpKinds.has(op.kind);
}
/**
 * Create an `ElementStartOp`.
 */
export function createElementStartOp(tag, xref, namespace, i18nPlaceholder, sourceSpan) {
    return {
        kind: OpKind.ElementStart,
        xref,
        tag,
        handle: new SlotHandle(),
        attributes: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `TemplateOp`.
 */
export function createTemplateOp(xref, tag, functionNameSuffix, namespace, i18nPlaceholder, sourceSpan) {
    return {
        kind: OpKind.Template,
        xref,
        attributes: null,
        tag,
        handle: new SlotHandle(),
        functionNameSuffix,
        decls: null,
        vars: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
export function createRepeaterCreateOp(primaryView, emptyView, tag, track, varNames, sourceSpan) {
    return {
        kind: OpKind.RepeaterCreate,
        attributes: null,
        xref: primaryView,
        handle: new SlotHandle(),
        emptyView,
        track,
        trackByFn: null,
        tag,
        functionNameSuffix: 'For',
        namespace: Namespace.HTML,
        nonBindable: false,
        localRefs: [],
        decls: null,
        vars: null,
        varNames,
        usesComponentInstance: false,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
        numSlotsUsed: emptyView === null ? 2 : 3,
    };
}
/**
 * Create an `ElementEndOp`.
 */
export function createElementEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.ElementEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createDisableBindingsOp(xref) {
    return {
        kind: OpKind.DisableBindings,
        xref,
        ...NEW_OP,
    };
}
export function createEnableBindingsOp(xref) {
    return {
        kind: OpKind.EnableBindings,
        xref,
        ...NEW_OP,
    };
}
/**
 * Create a `TextOp`.
 */
export function createTextOp(xref, initialValue, sourceSpan) {
    return {
        kind: OpKind.Text,
        xref,
        handle: new SlotHandle(),
        initialValue,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `ListenerOp`. Host bindings reuse all the listener logic.
 */
export function createListenerOp(target, targetSlot, name, tag, animationPhase, hostListener, sourceSpan) {
    return {
        kind: OpKind.Listener,
        target,
        targetSlot,
        tag,
        hostListener,
        name,
        handlerOps: new OpList(),
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: animationPhase !== null,
        animationPhase: animationPhase,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createPipeOp(xref, slot, name) {
    return {
        kind: OpKind.Pipe,
        xref,
        handle: slot,
        name,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
export function createNamespaceOp(namespace) {
    return {
        kind: OpKind.Namespace,
        active: namespace,
        ...NEW_OP,
    };
}
export function createProjectionDefOp(def) {
    return {
        kind: OpKind.ProjectionDef,
        def,
        ...NEW_OP,
    };
}
export function createProjectionOp(xref, selector, sourceSpan) {
    return {
        kind: OpKind.Projection,
        xref,
        handle: new SlotHandle(),
        selector,
        projectionSlotIndex: 0,
        attributes: [],
        localRefs: [],
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `ExtractedAttributeOp`.
 */
export function createExtractedAttributeOp(target, bindingKind, name, expression) {
    return {
        kind: OpKind.ExtractedAttribute,
        target,
        bindingKind,
        name,
        expression,
        ...NEW_OP,
    };
}
export function createDeferOp(xref, main, mainSlot, metadata, sourceSpan) {
    return {
        kind: OpKind.Defer,
        xref,
        handle: new SlotHandle(),
        mainView: main,
        mainSlot,
        loadingView: null,
        loadingSlot: null,
        loadingConfig: null,
        loadingMinimumTime: null,
        loadingAfterTime: null,
        placeholderView: null,
        placeholderSlot: null,
        placeholderConfig: null,
        placeholderMinimumTime: null,
        errorView: null,
        errorSlot: null,
        metadata,
        resolverFn: null,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: 2,
    };
}
export function createDeferOnOp(defer, trigger, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferOn,
        defer,
        trigger,
        prefetch,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create an `ExtractedMessageOp`.
 */
export function createI18nMessageOp(xref, i18nBlock, message, messagePlaceholder, params, postprocessingParams, needsPostprocessing) {
    return {
        kind: OpKind.I18nMessage,
        xref,
        i18nBlock,
        message,
        messagePlaceholder,
        params,
        postprocessingParams,
        needsPostprocessing,
        subMessages: [],
        ...NEW_OP,
    };
}
/**
 * Create an `I18nStartOp`.
 */
export function createI18nStartOp(xref, message, root) {
    return {
        kind: OpKind.I18nStart,
        xref,
        handle: new SlotHandle(),
        root: root ?? xref,
        message,
        messageIndex: null,
        subTemplateIndex: null,
        context: null,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `I18nEndOp`.
 */
export function createI18nEndOp(xref) {
    return {
        kind: OpKind.I18nEnd,
        xref,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU start op.
 */
export function createIcuStartOp(xref, message, messagePlaceholder, sourceSpan) {
    return {
        kind: OpKind.IcuStart,
        xref,
        message,
        messagePlaceholder,
        context: null,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU end op.
 */
export function createIcuEndOp(xref) {
    return {
        kind: OpKind.IcuEnd,
        xref,
        ...NEW_OP,
    };
}
export function createI18nContextOp(xref, i18nBlock, message, sourceSpan) {
    return {
        kind: OpKind.I18nContext,
        xref,
        i18nBlock,
        message,
        sourceSpan,
        params: new Map(),
        postprocessingParams: new Map(),
        ...NEW_OP,
    };
}
export function literalOrArrayLiteral(value) {
    if (Array.isArray(value)) {
        return o.literalArr(value.map(literalOrArrayLiteral));
    }
    return o.literal(value, o.INFERRED_TYPE);
}
//# sourceMappingURL=data:application/json;base64,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