/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../output/output_ast';
import { Identifiers as R3 } from './r3_identifiers';
import { jitOnlyGuardedExpression, refsToArray } from './util';
import { DefinitionMap } from './view/util';
/**
 * How the selector scope of an NgModule (its declarations, imports, and exports) should be emitted
 * as a part of the NgModule definition.
 */
export var R3SelectorScopeMode;
(function (R3SelectorScopeMode) {
    /**
     * Emit the declarations inline into the module definition.
     *
     * This option is useful in certain contexts where it's known that JIT support is required. The
     * tradeoff here is that this emit style prevents directives and pipes from being tree-shaken if
     * they are unused, but the NgModule is used.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["Inline"] = 0] = "Inline";
    /**
     * Emit the declarations using a side effectful function call, `ɵɵsetNgModuleScope`, that is
     * guarded with the `ngJitMode` flag.
     *
     * This form of emit supports JIT and can be optimized away if the `ngJitMode` flag is set to
     * false, which allows unused directives and pipes to be tree-shaken.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["SideEffect"] = 1] = "SideEffect";
    /**
     * Don't generate selector scopes at all.
     *
     * This is useful for contexts where JIT support is known to be unnecessary.
     */
    R3SelectorScopeMode[R3SelectorScopeMode["Omit"] = 2] = "Omit";
})(R3SelectorScopeMode || (R3SelectorScopeMode = {}));
/**
 * The type of the NgModule meta data.
 * - Global: Used for full and partial compilation modes which mainly includes R3References.
 * - Local: Used for the local compilation mode which mainly includes the raw expressions as appears
 * in the NgModule decorator.
 */
export var R3NgModuleMetadataKind;
(function (R3NgModuleMetadataKind) {
    R3NgModuleMetadataKind[R3NgModuleMetadataKind["Global"] = 0] = "Global";
    R3NgModuleMetadataKind[R3NgModuleMetadataKind["Local"] = 1] = "Local";
})(R3NgModuleMetadataKind || (R3NgModuleMetadataKind = {}));
/**
 * Construct an `R3NgModuleDef` for the given `R3NgModuleMetadata`.
 */
export function compileNgModule(meta) {
    const statements = [];
    const definitionMap = new DefinitionMap();
    definitionMap.set('type', meta.type.value);
    // Assign bootstrap definition. In local compilation mode (i.e., for
    // `R3NgModuleMetadataKind.LOCAL`) we assign the bootstrap field using the runtime
    // `ɵɵsetNgModuleScope`.
    if (meta.kind === R3NgModuleMetadataKind.Global && meta.bootstrap.length > 0) {
        definitionMap.set('bootstrap', refsToArray(meta.bootstrap, meta.containsForwardDecls));
    }
    if (meta.selectorScopeMode === R3SelectorScopeMode.Inline) {
        // If requested to emit scope information inline, pass the `declarations`, `imports` and
        // `exports` to the `ɵɵdefineNgModule()` call directly.
        if (meta.declarations.length > 0) {
            definitionMap.set('declarations', refsToArray(meta.declarations, meta.containsForwardDecls));
        }
        if (meta.imports.length > 0) {
            definitionMap.set('imports', refsToArray(meta.imports, meta.containsForwardDecls));
        }
        if (meta.exports.length > 0) {
            definitionMap.set('exports', refsToArray(meta.exports, meta.containsForwardDecls));
        }
    }
    else if (meta.selectorScopeMode === R3SelectorScopeMode.SideEffect) {
        // In this mode, scope information is not passed into `ɵɵdefineNgModule` as it
        // would prevent tree-shaking of the declarations, imports and exports references. Instead, it's
        // patched onto the NgModule definition with a `ɵɵsetNgModuleScope` call that's guarded by the
        // `ngJitMode` flag.
        const setNgModuleScopeCall = generateSetNgModuleScopeCall(meta);
        if (setNgModuleScopeCall !== null) {
            statements.push(setNgModuleScopeCall);
        }
    }
    else {
        // Selector scope emit was not requested, so skip it.
    }
    if (meta.schemas !== null && meta.schemas.length > 0) {
        definitionMap.set('schemas', o.literalArr(meta.schemas.map(ref => ref.value)));
    }
    if (meta.id !== null) {
        definitionMap.set('id', meta.id);
        // Generate a side-effectful call to register this NgModule by its id, as per the semantics of
        // NgModule ids.
        statements.push(o.importExpr(R3.registerNgModuleType).callFn([meta.type.value, meta.id]).toStmt());
    }
    const expression = o.importExpr(R3.defineNgModule).callFn([definitionMap.toLiteralMap()], undefined, true);
    const type = createNgModuleType(meta);
    return { expression, type, statements };
}
/**
 * This function is used in JIT mode to generate the call to `ɵɵdefineNgModule()` from a call to
 * `ɵɵngDeclareNgModule()`.
 */
export function compileNgModuleDeclarationExpression(meta) {
    const definitionMap = new DefinitionMap();
    definitionMap.set('type', new o.WrappedNodeExpr(meta.type));
    if (meta.bootstrap !== undefined) {
        definitionMap.set('bootstrap', new o.WrappedNodeExpr(meta.bootstrap));
    }
    if (meta.declarations !== undefined) {
        definitionMap.set('declarations', new o.WrappedNodeExpr(meta.declarations));
    }
    if (meta.imports !== undefined) {
        definitionMap.set('imports', new o.WrappedNodeExpr(meta.imports));
    }
    if (meta.exports !== undefined) {
        definitionMap.set('exports', new o.WrappedNodeExpr(meta.exports));
    }
    if (meta.schemas !== undefined) {
        definitionMap.set('schemas', new o.WrappedNodeExpr(meta.schemas));
    }
    if (meta.id !== undefined) {
        definitionMap.set('id', new o.WrappedNodeExpr(meta.id));
    }
    return o.importExpr(R3.defineNgModule).callFn([definitionMap.toLiteralMap()]);
}
export function createNgModuleType(meta) {
    if (meta.kind === R3NgModuleMetadataKind.Local) {
        return new o.ExpressionType(meta.type.value);
    }
    const { type: moduleType, declarations, exports, imports, includeImportTypes, publicDeclarationTypes } = meta;
    return new o.ExpressionType(o.importExpr(R3.NgModuleDeclaration, [
        new o.ExpressionType(moduleType.type),
        publicDeclarationTypes === null ? tupleTypeOf(declarations) :
            tupleOfTypes(publicDeclarationTypes),
        includeImportTypes ? tupleTypeOf(imports) : o.NONE_TYPE,
        tupleTypeOf(exports),
    ]));
}
/**
 * Generates a function call to `ɵɵsetNgModuleScope` with all necessary information so that the
 * transitive module scope can be computed during runtime in JIT mode. This call is marked pure
 * such that the references to declarations, imports and exports may be elided causing these
 * symbols to become tree-shakeable.
 */
function generateSetNgModuleScopeCall(meta) {
    const scopeMap = new DefinitionMap();
    if (meta.kind === R3NgModuleMetadataKind.Global) {
        if (meta.declarations.length > 0) {
            scopeMap.set('declarations', refsToArray(meta.declarations, meta.containsForwardDecls));
        }
    }
    else {
        if (meta.declarationsExpression) {
            scopeMap.set('declarations', meta.declarationsExpression);
        }
    }
    if (meta.kind === R3NgModuleMetadataKind.Global) {
        if (meta.imports.length > 0) {
            scopeMap.set('imports', refsToArray(meta.imports, meta.containsForwardDecls));
        }
    }
    else {
        if (meta.importsExpression) {
            scopeMap.set('imports', meta.importsExpression);
        }
    }
    if (meta.kind === R3NgModuleMetadataKind.Global) {
        if (meta.exports.length > 0) {
            scopeMap.set('exports', refsToArray(meta.exports, meta.containsForwardDecls));
        }
    }
    else {
        if (meta.exportsExpression) {
            scopeMap.set('exports', meta.exportsExpression);
        }
    }
    if (meta.kind === R3NgModuleMetadataKind.Local && meta.bootstrapExpression) {
        scopeMap.set('bootstrap', meta.bootstrapExpression);
    }
    if (Object.keys(scopeMap.values).length === 0) {
        return null;
    }
    // setNgModuleScope(...)
    const fnCall = new o.InvokeFunctionExpr(
    /* fn */ o.importExpr(R3.setNgModuleScope), 
    /* args */ [meta.type.value, scopeMap.toLiteralMap()]);
    // (ngJitMode guard) && setNgModuleScope(...)
    const guardedCall = jitOnlyGuardedExpression(fnCall);
    // function() { (ngJitMode guard) && setNgModuleScope(...); }
    const iife = new o.FunctionExpr(
    /* params */ [], 
    /* statements */ [guardedCall.toStmt()]);
    // (function() { (ngJitMode guard) && setNgModuleScope(...); })()
    const iifeCall = new o.InvokeFunctionExpr(
    /* fn */ iife, 
    /* args */ []);
    return iifeCall.toStmt();
}
function tupleTypeOf(exp) {
    const types = exp.map(ref => o.typeofExpr(ref.type));
    return exp.length > 0 ? o.expressionType(o.literalArr(types)) : o.NONE_TYPE;
}
function tupleOfTypes(types) {
    const typeofTypes = types.map(type => o.typeofExpr(type));
    return types.length > 0 ? o.expressionType(o.literalArr(typeofTypes)) : o.NONE_TYPE;
}
//# sourceMappingURL=data:application/json;base64,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