/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { filter, map, take } from 'rxjs/operators';
import { NavigationCancel, NavigationEnd, NavigationError, NavigationSkipped } from '../events';
var NavigationResult;
(function (NavigationResult) {
    NavigationResult[NavigationResult["COMPLETE"] = 0] = "COMPLETE";
    NavigationResult[NavigationResult["FAILED"] = 1] = "FAILED";
    NavigationResult[NavigationResult["REDIRECTING"] = 2] = "REDIRECTING";
})(NavigationResult || (NavigationResult = {}));
/**
 * Performs the given action once the router finishes its next/current navigation.
 *
 * The navigation is considered complete under the following conditions:
 * - `NavigationCancel` event emits and the code is not `NavigationCancellationCode.Redirect` or
 * `NavigationCancellationCode.SupersededByNewNavigation`. In these cases, the
 * redirecting/superseding navigation must finish.
 * - `NavigationError`, `NavigationEnd`, or `NavigationSkipped` event emits
 */
export function afterNextNavigation(router, action) {
    router.events
        .pipe(filter((e) => e instanceof NavigationEnd || e instanceof NavigationCancel ||
        e instanceof NavigationError || e instanceof NavigationSkipped), map(e => {
        if (e instanceof NavigationEnd || e instanceof NavigationSkipped) {
            return NavigationResult.COMPLETE;
        }
        const redirecting = e instanceof NavigationCancel ?
            (e.code === 0 /* NavigationCancellationCode.Redirect */ ||
                e.code === 1 /* NavigationCancellationCode.SupersededByNewNavigation */) :
            false;
        return redirecting ? NavigationResult.REDIRECTING : NavigationResult.FAILED;
    }), filter((result) => result !== NavigationResult.REDIRECTING), take(1))
        .subscribe(() => {
        action();
    });
}
//# sourceMappingURL=data:application/json;base64,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