/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { EMPTY, from, of, throwError } from 'rxjs';
import { catchError, concatMap, first, map, mapTo, mergeMap, takeLast, tap } from 'rxjs/operators';
import { getInherited, hasStaticTitle } from '../router_state';
import { RouteTitleKey } from '../shared';
import { getDataKeys, wrapIntoObservable } from '../utils/collection';
import { getClosestRouteInjector } from '../utils/config';
import { getTokenOrFunctionIdentity } from '../utils/preactivation';
import { isEmptyError } from '../utils/type_guards';
export function resolveData(paramsInheritanceStrategy, injector) {
    return mergeMap(t => {
        const { targetSnapshot, guards: { canActivateChecks } } = t;
        if (!canActivateChecks.length) {
            return of(t);
        }
        const routesWithResolversToRun = canActivateChecks.map(check => check.route);
        const routesWithResolversSet = new Set(routesWithResolversToRun);
        const routesNeedingDataUpdates = 
        // List all ActivatedRoutes in an array, starting from the parent of the first route to run
        // resolvers. We go from the parent because the first route might have siblings that also
        // run resolvers.
        flattenRouteTree(routesWithResolversToRun[0].parent)
            // Remove the parent from the list -- we do not need to recompute its inherited data
            // because no resolvers at or above it run.
            .slice(1);
        let routesProcessed = 0;
        return from(routesNeedingDataUpdates)
            .pipe(concatMap(route => {
            if (routesWithResolversSet.has(route)) {
                return runResolve(route, targetSnapshot, paramsInheritanceStrategy, injector);
            }
            else {
                route.data = getInherited(route, route.parent, paramsInheritanceStrategy).resolve;
                return of(void 0);
            }
        }), tap(() => routesProcessed++), takeLast(1), mergeMap(_ => routesProcessed === routesNeedingDataUpdates.length ? of(t) : EMPTY));
    });
}
/**
 *  Returns the `ActivatedRouteSnapshot` tree as an array, using DFS to traverse the route tree.
 */
function flattenRouteTree(route) {
    const descendants = route.children.map(child => flattenRouteTree(child)).flat();
    return [route, ...descendants];
}
function runResolve(futureARS, futureRSS, paramsInheritanceStrategy, injector) {
    const config = futureARS.routeConfig;
    const resolve = futureARS._resolve;
    if (config?.title !== undefined && !hasStaticTitle(config)) {
        resolve[RouteTitleKey] = config.title;
    }
    return resolveNode(resolve, futureARS, futureRSS, injector).pipe(map((resolvedData) => {
        futureARS._resolvedData = resolvedData;
        futureARS.data = getInherited(futureARS, futureARS.parent, paramsInheritanceStrategy).resolve;
        return null;
    }));
}
function resolveNode(resolve, futureARS, futureRSS, injector) {
    const keys = getDataKeys(resolve);
    if (keys.length === 0) {
        return of({});
    }
    const data = {};
    return from(keys).pipe(mergeMap(key => getResolver(resolve[key], futureARS, futureRSS, injector)
        .pipe(first(), tap((value) => {
        data[key] = value;
    }))), takeLast(1), mapTo(data), catchError((e) => isEmptyError(e) ? EMPTY : throwError(e)));
}
function getResolver(injectionToken, futureARS, futureRSS, injector) {
    const closestInjector = getClosestRouteInjector(futureARS) ?? injector;
    const resolver = getTokenOrFunctionIdentity(injectionToken, closestInjector);
    const resolverValue = resolver.resolve ?
        resolver.resolve(futureARS, futureRSS) :
        closestInjector.runInContext(() => resolver(futureARS, futureRSS));
    return wrapIntoObservable(resolverValue);
}
//# sourceMappingURL=data:application/json;base64,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