/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { isForwardRef, resolveForwardRef } from '../di/forward_ref';
import { flatten } from '../util/array_utils';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY } from '../util/empty';
import { extractDefListOrFactory, getNgModuleDef } from './definition';
import { depsTracker } from './deps_tracker/deps_tracker';
import { isModuleWithProviders } from './jit/util';
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = extractDefListOrFactory(directives, /* pipeDef */ false);
    def.pipeDefs = extractDefListOrFactory(pipes, /* pipeDef */ true);
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = convertToTypeArray(scope.declarations || EMPTY_ARRAY);
        ngModuleDef.imports = convertToTypeArray(scope.imports || EMPTY_ARRAY);
        ngModuleDef.exports = convertToTypeArray(scope.exports || EMPTY_ARRAY);
        if (scope.bootstrap) {
            // This only happens in local compilation mode.
            ngModuleDef.bootstrap = convertToTypeArray(scope.bootstrap);
        }
        depsTracker.registerNgModule(type, scope);
    });
}
function convertToTypeArray(values) {
    if (typeof values === 'function') {
        return values;
    }
    const flattenValues = flatten(values);
    if (flattenValues.some(isForwardRef)) {
        return () => flattenValues.map(resolveForwardRef).map(maybeUnwrapModuleWithProviders);
    }
    else {
        return flattenValues.map(maybeUnwrapModuleWithProviders);
    }
}
function maybeUnwrapModuleWithProviders(value) {
    return isModuleWithProviders(value) ? value.ngModule : value;
}
//# sourceMappingURL=data:application/json;base64,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