/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as ir from '../../ir';
function kindTest(kind) {
    return (op) => op.kind === kind;
}
function kindWithInterpolationTest(kind, interpolation) {
    return (op) => {
        return op.kind === kind && interpolation === op.expression instanceof ir.Interpolation;
    };
}
/**
 * Defines the groups based on `OpKind` that ops will be divided into, for the various create
 * op kinds. Ops will be collected into groups, then optionally transformed, before recombining
 * the groups in the order defined here.
 */
const CREATE_ORDERING = [
    { test: op => op.kind === ir.OpKind.Listener && op.hostListener && op.isAnimationListener },
    { test: op => op.kind === ir.OpKind.Listener && !(op.hostListener && op.isAnimationListener) },
];
/**
 * Defines the groups based on `OpKind` that ops will be divided into, for the various update
 * op kinds.
 */
const UPDATE_ORDERING = [
    { test: kindWithInterpolationTest(ir.OpKind.HostProperty, true) },
    { test: kindWithInterpolationTest(ir.OpKind.HostProperty, false) },
    { test: kindTest(ir.OpKind.StyleMap), transform: keepLast },
    { test: kindTest(ir.OpKind.ClassMap), transform: keepLast },
    { test: kindTest(ir.OpKind.StyleProp) },
    { test: kindTest(ir.OpKind.ClassProp) },
    { test: kindWithInterpolationTest(ir.OpKind.Property, true) },
    { test: kindWithInterpolationTest(ir.OpKind.Attribute, true) },
    { test: kindWithInterpolationTest(ir.OpKind.Property, false) },
    { test: kindWithInterpolationTest(ir.OpKind.Attribute, false) },
];
/**
 * The set of all op kinds we handle in the reordering phase.
 */
const handledOpKinds = new Set([
    ir.OpKind.Listener, ir.OpKind.StyleMap, ir.OpKind.ClassMap, ir.OpKind.StyleProp,
    ir.OpKind.ClassProp, ir.OpKind.Property, ir.OpKind.HostProperty, ir.OpKind.Attribute
]);
/**
 * Many type of operations have ordering constraints that must be respected. For example, a
 * `ClassMap` instruction must be ordered after a `StyleMap` instruction, in order to have
 * predictable semantics that match TemplateDefinitionBuilder and don't break applications.
 */
export function orderOps(job) {
    for (const unit of job.units) {
        // First, we pull out ops that need to be ordered. Then, when we encounter an op that shouldn't
        // be reordered, put the ones we've pulled so far back in the correct order. Finally, if we
        // still have ops pulled at the end, put them back in the correct order.
        // Create mode:
        orderWithin(unit.create, CREATE_ORDERING);
        // Update mode:
        orderWithin(unit.update, UPDATE_ORDERING);
    }
}
/**
 * Order all the ops within the specified group.
 */
function orderWithin(opList, ordering) {
    let opsToOrder = [];
    // Only reorder ops that target the same xref; do not mix ops that target different xrefs.
    let firstTargetInGroup = null;
    for (const op of opList) {
        const currentTarget = ir.hasDependsOnSlotContextTrait(op) ? op.target : null;
        if (!handledOpKinds.has(op.kind) ||
            (currentTarget !== firstTargetInGroup &&
                (firstTargetInGroup !== null && currentTarget !== null))) {
            ir.OpList.insertBefore(reorder(opsToOrder, ordering), op);
            opsToOrder = [];
            firstTargetInGroup = null;
        }
        if (handledOpKinds.has(op.kind)) {
            opsToOrder.push(op);
            ir.OpList.remove(op);
            firstTargetInGroup = currentTarget ?? firstTargetInGroup;
        }
    }
    opList.push(reorder(opsToOrder, ordering));
}
/**
 * Reorders the given list of ops according to the ordering defined by `ORDERING`.
 */
function reorder(ops, ordering) {
    // Break the ops list into groups based on OpKind.
    const groups = Array.from(ordering, () => new Array());
    for (const op of ops) {
        const groupIndex = ordering.findIndex(o => o.test(op));
        groups[groupIndex].push(op);
    }
    // Reassemble the groups into a single list, in the correct order.
    return groups.flatMap((group, i) => {
        const transform = ordering[i].transform;
        return transform ? transform(group) : group;
    });
}
/**
 * Keeps only the last op in a list of ops.
 */
function keepLast(ops) {
    return ops.slice(ops.length - 1);
}
//# sourceMappingURL=data:application/json;base64,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