/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as ir from '../../ir';
const STYLE_DOT = 'style.';
const CLASS_DOT = 'class.';
const STYLE_BANG = 'style!';
const CLASS_BANG = 'class!';
const BANG_IMPORTANT = '!important';
/**
 * Host bindings are compiled using a different parser entrypoint, and are parsed quite differently
 * as a result. Therefore, we need to do some extra parsing for host style properties, as compared
 * to non-host style properties.
 * TODO: Unify host bindings and non-host bindings in the parser.
 */
export function parseHostStyleProperties(job) {
    for (const op of job.root.update) {
        if (op.kind !== ir.OpKind.Binding) {
            continue;
        }
        if (op.name.endsWith(BANG_IMPORTANT)) {
            // Delete any `!important` suffixes from the binding name.
            op.name = op.name.substring(0, op.name.length - BANG_IMPORTANT.length);
        }
        if (op.name.startsWith(STYLE_DOT)) {
            op.bindingKind = ir.BindingKind.StyleProperty;
            op.name = op.name.substring(STYLE_DOT.length);
            if (isCssCustomProperty(op.name)) {
                op.name = hyphenate(op.name);
            }
            const { property, suffix } = parseProperty(op.name);
            op.name = property;
            op.unit = suffix;
        }
        else if (op.name.startsWith(STYLE_BANG)) {
            op.bindingKind = ir.BindingKind.StyleProperty;
            op.name = 'style';
        }
        else if (op.name.startsWith(CLASS_DOT)) {
            op.bindingKind = ir.BindingKind.ClassName;
            op.name = parseProperty(op.name.substring(CLASS_DOT.length)).property;
        }
        else if (op.name.startsWith(CLASS_BANG)) {
            op.bindingKind = ir.BindingKind.ClassName;
            op.name = parseProperty(op.name.substring(CLASS_BANG.length)).property;
        }
    }
}
/**
 * Checks whether property name is a custom CSS property.
 * See: https://www.w3.org/TR/css-variables-1
 */
function isCssCustomProperty(name) {
    return name.startsWith('--');
}
function hyphenate(value) {
    return value
        .replace(/[a-z][A-Z]/g, v => {
        return v.charAt(0) + '-' + v.charAt(1);
    })
        .toLowerCase();
}
function parseProperty(name) {
    const overrideIndex = name.indexOf('!important');
    if (overrideIndex !== -1) {
        name = overrideIndex > 0 ? name.substring(0, overrideIndex) : '';
    }
    let suffix = null;
    let property = name;
    const unitIndex = name.lastIndexOf('.');
    if (unitIndex > 0) {
        suffix = name.slice(unitIndex + 1);
        property = name.substring(0, unitIndex);
    }
    return { property, suffix };
}
//# sourceMappingURL=data:application/json;base64,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