/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * The escape sequence used indicate message param values.
 */
const ESCAPE = '\uFFFD';
/**
 * Marker used to indicate an element tag.
 */
const ELEMENT_MARKER = '#';
/**
 * Marker used to indicate a template tag.
 */
const TEMPLATE_MARKER = '*';
/**
 * Marker used to indicate closing of an element or template tag.
 */
const TAG_CLOSE_MARKER = '/';
/**
 * Marker used to indicate the sub-template context.
 */
const CONTEXT_MARKER = ':';
/**
 * Marker used to indicate the start of a list of values.
 */
const LIST_START_MARKER = '[';
/**
 * Marker used to indicate the end of a list of values.
 */
const LIST_END_MARKER = ']';
/**
 * Delimiter used to separate multiple values in a list.
 */
const LIST_DELIMITER = '|';
/**
 * Formats the param maps on extracted message ops into a maps of `Expression` objects that can be
 * used in the final output.
 */
export function extractI18nMessages(job) {
    // Save the i18n context ops for later use.
    const i18nContexts = new Map();
    // Record which contexts represent i18n blocks (any other contexts are assumed to have been
    // created from ICUs).
    const i18nBlockContexts = new Set();
    for (const unit of job.units) {
        for (const op of unit.create) {
            switch (op.kind) {
                case ir.OpKind.I18nContext:
                    i18nContexts.set(op.xref, op);
                    break;
                case ir.OpKind.I18nStart:
                    i18nBlockContexts.add(op.context);
                    break;
            }
        }
    }
    // Extract messages from root i18n blocks.
    const i18nBlockMessages = new Map();
    for (const unit of job.units) {
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.I18nStart && op.xref === op.root) {
                if (!op.context) {
                    throw Error('I18n start op should have its context set.');
                }
                const i18nMessageOp = createI18nMessage(job, i18nContexts.get(op.context));
                i18nBlockMessages.set(op.xref, i18nMessageOp);
                unit.create.push(i18nMessageOp);
            }
        }
    }
    // Extract messages from ICUs with their own sub-context.
    for (const unit of job.units) {
        for (const op of unit.create) {
            switch (op.kind) {
                case ir.OpKind.IcuStart:
                    if (!op.context) {
                        throw Error('ICU op should have its context set.');
                    }
                    if (!i18nBlockContexts.has(op.context)) {
                        const i18nContext = i18nContexts.get(op.context);
                        const subMessage = createI18nMessage(job, i18nContext, op.messagePlaceholder);
                        unit.create.push(subMessage);
                        const parentMessage = i18nBlockMessages.get(i18nContext.i18nBlock);
                        parentMessage?.subMessages.push(subMessage.xref);
                    }
                    ir.OpList.remove(op);
                    break;
                case ir.OpKind.IcuEnd:
                    ir.OpList.remove(op);
                    break;
            }
        }
    }
}
/**
 * Create an i18n message op from an i18n context op.
 */
function createI18nMessage(job, context, messagePlaceholder) {
    let needsPostprocessing = context.postprocessingParams.size > 0;
    for (const values of context.params.values()) {
        if (values.length > 1) {
            needsPostprocessing = true;
        }
    }
    return ir.createI18nMessageOp(job.allocateXrefId(), context.i18nBlock, context.message, messagePlaceholder ?? null, formatParams(context.params), formatParams(context.postprocessingParams), needsPostprocessing);
}
/**
 * Formats a map of `I18nParamValue[]` values into a map of `Expression` values.
 */
function formatParams(params) {
    const result = new Map();
    for (const [placeholder, placeholderValues] of params) {
        const serializedValues = formatParamValues(placeholderValues);
        if (serializedValues !== null) {
            result.set(placeholder, o.literal(formatParamValues(placeholderValues)));
        }
    }
    return result;
}
/**
 * Formats an `I18nParamValue[]` into a string (or null for empty array).
 */
function formatParamValues(values) {
    if (values.length === 0) {
        return null;
    }
    const serializedValues = values.map(value => formatValue(value));
    return serializedValues.length === 1 ?
        serializedValues[0] :
        `${LIST_START_MARKER}${serializedValues.join(LIST_DELIMITER)}${LIST_END_MARKER}`;
}
/**
 * Formats a single `I18nParamValue` into a string
 */
function formatValue(value) {
    // If there are no special flags, just return the raw value.
    if (value.flags === ir.I18nParamValueFlags.None) {
        return `${value.value}`;
    }
    let tagMarker = '';
    let closeMarker = '';
    if (value.flags & ir.I18nParamValueFlags.ElementTag) {
        tagMarker = ELEMENT_MARKER;
    }
    else if (value.flags & ir.I18nParamValueFlags.TemplateTag) {
        tagMarker = TEMPLATE_MARKER;
    }
    if (tagMarker !== '') {
        closeMarker = value.flags & ir.I18nParamValueFlags.CloseTag ? TAG_CLOSE_MARKER : '';
    }
    const context = value.subTemplateIndex === null ? '' : `${CONTEXT_MARKER}${value.subTemplateIndex}`;
    // Self-closing tags use a special form that concatenates the start and close tag values.
    if ((value.flags & ir.I18nParamValueFlags.OpenTag) &&
        (value.flags & ir.I18nParamValueFlags.CloseTag)) {
        return `${ESCAPE}${tagMarker}${value.value}${context}${ESCAPE}${ESCAPE}${closeMarker}${tagMarker}${value.value}${context}${ESCAPE}`;
    }
    return `${ESCAPE}${closeMarker}${tagMarker}${value.value}${context}${ESCAPE}`;
}
//# sourceMappingURL=data:application/json;base64,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