/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { OpKind } from '../enums';
import { TRAIT_CONSUMES_VARS, TRAIT_DEPENDS_ON_SLOT_CONTEXT } from '../traits';
import { NEW_OP } from './shared';
/**
 * Create an `InterpolationTextOp`.
 */
export function createInterpolateTextOp(xref, interpolation, i18nPlaceholders, sourceSpan) {
    return {
        kind: OpKind.InterpolateText,
        target: xref,
        interpolation,
        i18nPlaceholders,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
export class Interpolation {
    constructor(strings, expressions) {
        this.strings = strings;
        this.expressions = expressions;
    }
}
/**
 * Create a `BindingOp`, not yet transformed into a particular type of binding.
 */
export function createBindingOp(target, kind, name, expression, unit, securityContext, isTextAttribute, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Binding,
        bindingKind: kind,
        target,
        name,
        expression,
        unit,
        securityContext,
        isTextAttribute,
        isTemplate,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `PropertyOp`.
 */
export function createPropertyOp(target, name, expression, isAnimationTrigger, securityContext, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Property,
        target,
        name,
        expression,
        isAnimationTrigger,
        securityContext,
        sanitizer: null,
        isTemplate,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StylePropOp`. */
export function createStylePropOp(xref, name, expression, unit, sourceSpan) {
    return {
        kind: OpKind.StyleProp,
        target: xref,
        name,
        expression,
        unit,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassPropOp`.
 */
export function createClassPropOp(xref, name, expression, sourceSpan) {
    return {
        kind: OpKind.ClassProp,
        target: xref,
        name,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StyleMapOp`. */
export function createStyleMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.StyleMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassMapOp`.
 */
export function createClassMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.ClassMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AttributeOp`.
 */
export function createAttributeOp(target, name, expression, securityContext, isTextAttribute, isTemplate, sourceSpan) {
    return {
        kind: OpKind.Attribute,
        target,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isTextAttribute,
        isTemplate,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AdvanceOp`.
 */
export function createAdvanceOp(delta, sourceSpan) {
    return {
        kind: OpKind.Advance,
        delta,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a conditional op, which will display an embedded view according to a condtion.
 */
export function createConditionalOp(target, targetSlot, test, conditions, sourceSpan) {
    return {
        kind: OpKind.Conditional,
        target,
        targetSlot,
        test,
        conditions,
        processed: null,
        sourceSpan,
        contextValue: null,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
    };
}
export function createRepeaterOp(repeaterCreate, targetSlot, collection, sourceSpan) {
    return {
        kind: OpKind.Repeater,
        target: repeaterCreate,
        targetSlot,
        collection,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
export function createDeferWhenOp(target, expr, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferWhen,
        target,
        expr,
        prefetch,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
/**
 * Create an i18n expression op.
 */
export function createI18nExpressionOp(context, target, handle, expression, i18nPlaceholder, resolutionTime, sourceSpan) {
    return {
        kind: OpKind.I18nExpression,
        context,
        target,
        handle,
        expression,
        i18nPlaceholder,
        resolutionTime,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_VARS,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
/**
 *Creates an op to apply i18n expression ops
 */
export function createI18nApplyOp(target, handle, sourceSpan) {
    return {
        kind: OpKind.I18nApply,
        target,
        handle,
        sourceSpan,
        ...NEW_OP,
    };
}
//# sourceMappingURL=data:application/json;base64,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